/************************************************************************
 * NAME:	tslist.c()
 *
 * DESCR:	Implements the track/sector list handling.
 *
 * NOTES:	- a track/sector (ts) list in this context refers to the
 *		  structure as it exists in the first "block" of a file
 *		  referenced by the catalog.  This includes the pointer
 *		  to the next ts as well as the sector offset.
 ************************************************************************/
#include "ados.h"
#include "standard.h"

/************************************************************************
 * NAME:	ados_tslist_create()
 *
 * DESCR:	Creates a new tslist structure with the given number of
 *		slots for the tslist.
 *
 * ARGS:	
 *
 * RETURNS:	TRUE if things went OK, FALSE otherwise.
 *
 * NOTES:	
 ************************************************************************/
ados_tslist *
ados_tslist_create(int size)
{
    ados_tslist	*retlist = NULL;

    retlist = (ados_tslist *)malloc(sizeof(ados_tslist));

    if (retlist != NULL) {
	retlist->list = (ados_ts_ref *)malloc(size*sizeof(ados_ts_ref));
	if (retlist->list == NULL) {
	    free(retlist);
	    retlist = NULL;
	}
    }

    retlist->count = size;
    retlist->cursor = 0;

    return(retlist);
}

void ados_tslist_destroy(ados_tslist *tslist)
{
    if (tslist) {
	free(tslist->list);
    }
    free(tslist);
}

/************************************************************************
 * NAME:	ados_tslist_write()
 *
 * DESCR:	Scribble the tslist back out to the disk.
 *
 * ARGS:	
 *
 * RETURNS:	
 *
 * NOTES:	
 ************************************************************************/
int
ados_tslist_write(struct adosfs *adosfs, ados_ts_ref *location, ados_tslist *tslist)
{
    unsigned char	*buffer;
    int			 sect_size = adosfs->vtoc->sector_size;
    int			 ts_count = adosfs->vtoc->max_ts;

    if (!tslist) {
	return(FALSE);
    }

    buffer = (unsigned char *)malloc(sect_size);
    if (!buffer) {
	return(FALSE);
    }

    buffer[0x01] = (unsigned char)tslist->next.track;
    buffer[0x02] = (unsigned char)tslist->next.sector;
    int_write(tslist->sector_offset,&(buffer[0x05]));

    {
	unsigned char 	*ptr = &(buffer[0x0c]);
	int		 i;

	for(i=0; i < ts_count; i++) {
	    *ptr++ = (unsigned char)tslist->list[i].track;
	    *ptr++ = (unsigned char)tslist->list[i].sector;
	}
    }

    if (!ados_putlsect(adosfs, location->track, location->sector, buffer)) {
	free(buffer);
	return(FALSE);
    }

    free(buffer);
    return(TRUE);
}


/************************************************************************
 * NAME:	ados_tslist_read()
 *
 * DESCR:	Reads a tslist from the filesystem at the given location.
 *
 * ARGS:	
 *
 * RETURNS:	good tslist pointer upon success, NULL upon failure.
 *
 * NOTES:	- tslist needs to be destroyed when done with it
 ************************************************************************/
ados_tslist *
ados_tslist_read(struct adosfs *adosfs, ados_ts_ref *location)
{
    unsigned char	*data;
    ados_tslist		*rettslist;
    int			 sect_size = adosfs->vtoc->sector_size;
    int			 ts_count = adosfs->vtoc->max_ts;
    int			 i;
    unsigned char	*ptr;

    rettslist = ados_tslist_create(ts_count);
    if (rettslist == NULL) {
	return(NULL);
    }
	
    data = (unsigned char *)malloc(sect_size);
    if (data == NULL) {
	ados_tslist_destroy(rettslist);
	return(NULL);
    }

    if (!ados_getlsect(adosfs, location->track, location->sector, data)) {
	ados_tslist_destroy(rettslist);
	free(data);
	return(NULL);
    }

    rettslist->next.track = (int)data[0x01];
    rettslist->next.sector = (int)data[0x02];
    rettslist->sector_offset = int_read(&(data[0x05]));

    ptr = &(data[0x0c]);
    for(i=0; i < ts_count; i++) {
	rettslist->list[i].track = (int)*ptr++;
	rettslist->list[i].sector = (int)*ptr++;
    }

    free(data);

    return(rettslist);
}
    
/************************************************************************
 * NAME:	ados_tslist_blank()
 *
 * DESCR:	Returns a new blank tslist (calls create).
 *
 * ARGS:	
 *
 * RETURNS:	
 *
 * NOTES:	
 ************************************************************************/
ados_tslist *
ados_tslist_blank(struct adosfs *adosfs, int offset)
{
    ados_tslist	*ret_tslist;
    int		 i;

    ret_tslist = ados_tslist_create(adosfs->vtoc->max_ts);
    if (ret_tslist != NULL) {
	ret_tslist->next.track = 0;
	ret_tslist->next.sector = 0;
	ret_tslist->sector_offset = offset;
	for (i=0; i < adosfs->vtoc->max_ts; i++) {
	    ret_tslist->list[i].track = 0;
	    ret_tslist->list[i].sector = 0;
	}
    }

    return(ret_tslist);
}
